<?php
namespace App\Http\Controllers;

use App\Models\NewsLanguage;
use App\Models\NewsLanguageSubscriber;
use App\Models\Setting;
use App\Models\Story;
use App\Models\Topic;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;

class WebStory extends Controller
{
    public function index(Request $request)
    {
        $title           = __('frontend-labels.web_stories.title');
        $topics          = Topic::all();
        $theme           = getTheme();
        $selectedTopicId = $request->query('topic');
        $userId          = Auth::user()->id ?? 0;

        if ($userId) {
            $subscribedLanguageIds = NewsLanguageSubscriber::where('user_id', $userId)->pluck('news_language_id');
        } else {
            $sessionLanguageId = session('selected_news_language');
            if ($sessionLanguageId) {
                $subscribedLanguageIds = collect([$sessionLanguageId]);
            } else {
                $defaultActiveLanguage = NewsLanguage::where('is_active', 1)->first();
                $subscribedLanguageIds = $defaultActiveLanguage ? collect([$defaultActiveLanguage->id]) : collect();
            }
        }

        $stories = Story::with(['story_slides', 'topic'])
            ->whereHas('story_slides')
            ->when($selectedTopicId, function (Builder $query) use ($selectedTopicId) {
                return $query->where('topic_id', $selectedTopicId);
            })
            ->when($subscribedLanguageIds->isNotEmpty(), function (Builder $query) use ($subscribedLanguageIds) {
                return $query->whereIn('news_language_id', $subscribedLanguageIds);
            })
            ->get();

        $filteredTopics = $topics->filter(function ($topic) use ($stories) {
            return $stories->contains('topic_id', $topic->id);
        });

        $dailyLimitReached        = false;
        $subscriptionLimitReached = false;

        $user = auth()->user();

        $freeTrialLimit = (int) Setting::where('name', 'free_trial_story_limit')->value('value');

        if ($user) {
            $subscription = $user->subscription;

            if ($subscription) {
                if ($subscription->hasReachedStoryLimits()) {
                    $subscriptionLimitReached = true;
                } else {
                    $subscription->incrementStoryCountWithValidation(0, 1);
                }
            } else {
                if ($freeTrialLimit !== -1) {
                    $sessionKey   = 'free_tier_story_count_' . $user->id;
                    $currentCount = session($sessionKey, 0);

                    if ($currentCount >= $freeTrialLimit) {
                        $dailyLimitReached = true;
                    } else {
                        session([$sessionKey => $currentCount + 1]);
                    }
                }
            }
        } else {
            if ($freeTrialLimit !== -1) {
                $sessionKey   = 'free_tier_story_count_guest';
                $currentCount = session($sessionKey, 0);

                if ($currentCount >= $freeTrialLimit) {
                    $dailyLimitReached = true;
                } else {
                    session([$sessionKey => $currentCount + 1]);
                }
            }
        }

        return view("front_end/{$theme}/pages/webstory", compact(
            'title',
            'filteredTopics',
            'theme',
            'stories',
            'selectedTopicId',
            'dailyLimitReached',
            'subscriptionLimitReached'
        ));
    }

    public function show(Topic $topic, Story $story)
    {
        $socialsettings = Setting::pluck('value', 'name');

        if ($story->topic_id !== $topic->id) {
            abort(404);
        }
        $userId = auth()->user()->id ?? 0;

        if ($userId) {
            $subscribedLanguageIds = NewsLanguageSubscriber::where('user_id', $userId)->pluck('news_language_id');
        } else {
            $sessionLanguageId = session('selected_news_language');
            if ($sessionLanguageId) {
                $subscribedLanguageIds = collect([$sessionLanguageId]);
            } else {
                $defaultActiveLanguage = NewsLanguage::where('is_active', 1)->first();
                $subscribedLanguageIds = $defaultActiveLanguage ? collect([$defaultActiveLanguage->id]) : collect();
            }
        }

        if ($subscribedLanguageIds->isNotEmpty() && ! $subscribedLanguageIds->contains($story->news_language_id)) {
            abort(404); // or redirect to webstories.index with message
        }

        $user                     = auth()->user();
        $dailyLimitReached        = false;
        $subscriptionLimitReached = false;

        $freeTrialLimit = (int) Setting::where('name', 'free_trial_story_limit')->value('value');

        if ($user) {
            $subscription = $user->subscription;

            if ($subscription) {
                if ($subscription->hasReachedStoryLimits()) {
                    return redirect()->route('webstories.index')
                        ->with('error', 'Maximum subscription limit reached for stories.');
                }
                $subscription->incrementStoryCountWithValidation(1, 0);
            } else {
                if ($freeTrialLimit !== -1) {
                    $sessionKey   = 'free_tier_story_count_' . $user->id;
                    $currentCount = session($sessionKey, 0);

                    if ($currentCount >= $freeTrialLimit) {
                        return redirect()->route('webstories.index')
                            ->with('error', 'Your free trial limit is complete; you can purchase a membership plan to continue using the service.');
                    }
                    session([$sessionKey => $currentCount + 1]);
                }
            }
        } else {
            if ($freeTrialLimit !== -1) {
                $sessionKey   = 'free_tier_story_count_guest';
                $currentCount = session($sessionKey, 0);

                if ($currentCount >= $freeTrialLimit) {
                    return redirect()->route('webstories.index')
                        ->with('error', 'Your free trial limit is complete; you can purchase a membership plan to continue using the service.');
                }
                session([$sessionKey => $currentCount + 1]);
            }
        }

        $this->storyCount($story);

        $nextStoryQuery = Story::with(['story_slides', 'topic'])
            ->where('topic_id', $topic->id)
            ->where('id', '>', $story->id)
            ->whereHas('story_slides');

        if ($subscribedLanguageIds->isNotEmpty()) {
            $nextStoryQuery->whereIn('news_language_id', $subscribedLanguageIds);
        }

        $nextStory = $nextStoryQuery->first();

        if (! $nextStory) {
            $fallbackQuery = Story::with(['story_slides', 'topic'])
                ->whereHas('story_slides')
                ->where('id', '!=', $story->id);
            if ($subscribedLanguageIds->isNotEmpty()) {
                $fallbackQuery->whereIn('news_language_id', $subscribedLanguageIds);
            }
            $nextStory = $fallbackQuery->first();
        }

        $animations = [];
        foreach ($story->story_slides as $slide) {
            $animations[$slide->id] = $slide->animation_details;
        }

        $theme = getTheme();

        return view("front_end/{$theme}/pages/webstory_slide", compact(
            'story',
            'theme',
            'nextStory',
            'animations',
            'socialsettings'
        ));
    }

    public function storyByTopic(Topic $topic)
    {
        $title  = __('frontend-labels.web_stories.title');
        $topics = Topic::all();

        $theme = getTheme();

        $stories = Story::with(['story_slides', 'topic'])
            ->where('topic_id', $topic->id)
            ->whereHas('story_slides')
            ->latest()
            ->paginate(12);

        $totalStories = $stories->total();

        return view("front_end/{$theme}/pages/webstory_by_topic", compact(
            'stories',
            'topic',
            'theme',
            'totalStories',
            'topics'
        ));
    }

    protected function storyCount($story)
    {
        $user_id    = Auth::user()->id ?? null;
        $cookieName = 'viewed_story_' . $story->id;

        // Check if the story has been viewed by checking the cookie
        if (! Cookie::has($cookieName)) {
            // Set a cookie to mark this story as viewed (expires in 15 days = 21600 minutes)
            Cookie::queue($cookieName, true, 21600);
            // Increment the story_count column
            $story->increment('story_count');
        }

        return $story;
    }
}
